// $Id: CCenterSlider.cpp,v 1.7 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CCenterSlider.hpp"
using Exponent::GUI::Controls::CCenterSlider;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CCenterSlider, CControl);

//	===========================================================================
CCenterSlider::CCenterSlider(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener) 
			 : CControl(root, uniqueId, area, listener)
			 , m_centerPoint(0)
			 , m_state(CRolloverButton::e_mouseOff)
			 , m_isVertical(true)
			 , m_isDragging(false)
			 , m_handleColour(CAlphaColour::CALPHACOLOUR_PURPLE)
			 , m_handleOverColour(CAlphaColour::CALPHACOLOUR_ORANGE)
			 , m_handleDownColour(CAlphaColour::CALPHACOLOUR_RED)
{
	EXPONENT_CLASS_CONSTRUCTION(CCenterSlider);

	// Are we vertical or horizontal?
	(m_area.getWidth() > m_area.getHeight()) ? m_isVertical = false : m_isVertical = true;

	// Default to center
	m_value = 1.0;

	if (m_isVertical)
	{
		m_centerPoint = m_area.getHeight() / 2;
		m_sliderArea.setRect(0, 0, m_area.getWidth(), m_centerPoint);
	}
	else
	{
		m_centerPoint = m_area.getWidth() / 2;
		m_sliderArea.setRect(m_centerPoint, 0, m_centerPoint, m_area.getHeight());
	}
}

//	===========================================================================
CCenterSlider::~CCenterSlider()
{
	EXPONENT_CLASS_DESTRUCTION(CCenterSlider);
}

//	===========================================================================
void CCenterSlider::setValue(const double value)
{
	// we want to set the slider top or bottom, depending on whether its greater than or less than zero
	if (value >= 0.0 && value <= 1.0)
	{
		m_value = value;
		if (m_isVertical)
		{
			if (value > 0.5)
			{
				const long theTop = (long)(m_centerPoint * (1.0 - ((value  - 0.5) * 2.0)));
				const long height = m_centerPoint - theTop;
				m_sliderArea.setTop(theTop);
				m_sliderArea.setHeight(height);
			}
			else if (value < 0.5)
			{
				m_sliderArea.setTop(m_centerPoint);
				m_sliderArea.setHeight(m_centerPoint - (long)((value * 2) * m_centerPoint));
			}
			else
			{
				// set to the same
				m_sliderArea.setTop(m_centerPoint);
				m_sliderArea.setHeight(1);
			}
		}
		else
		{
			if (value > 0.5)
			{
				m_sliderArea.setLeft(m_centerPoint);
				m_sliderArea.setWidth(m_centerPoint - (long)((1.0 - ((value - 0.5) * 2.0)) * m_centerPoint));
			}
			else if (value < 0.5)
			{
				const long theLeft = (long)(m_centerPoint * (value * 2.0));
				const long width   = m_centerPoint - theLeft;
				m_sliderArea.setLeft(theLeft);
				m_sliderArea.setWidth(width);
			}
			else
			{
				m_sliderArea.setLeft(m_centerPoint);
				m_sliderArea.setWidth(1);
			}
		}
	}
}

//	===========================================================================
void CCenterSlider::setColours(CAlphaColour frameColour, CAlphaColour backgroundColour, CAlphaColour handleColour, CAlphaColour handleOverColour, CAlphaColour handleDownColour)
{
	this->setDefaultControlColours(backgroundColour, frameColour);
	m_handleColour	   = handleColour;
	m_handleOverColour = handleOverColour;
	m_handleDownColour = handleDownColour;
}

//	===========================================================================
void CCenterSlider::handleLeftButtonDown(CMouseEvent &event)
{
	this->handleMovement(event);
	m_isDragging = true;
	m_rootControl->lockControl(this);
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this, event));
	}
	this->update();
}

//	===========================================================================
void CCenterSlider::handleLeftButtonUp(CMouseEvent &event)
{
	m_isDragging = false;
	m_rootControl->unlockControl();
	this->update();
}

//	===========================================================================
void CCenterSlider::handleMouseScroll(CMouseEvent &event)
{
	const double oldValue = m_value;
	double changeValue    = 0.0;

	if (event.getWheelMovementAmount() > 0.0)
	{
		(event.isShiftDown()) ? changeValue = 0.000625 : changeValue = 0.025;
	}
	else
	{
		(event.isShiftDown()) ? changeValue = -0.000625 : changeValue = -0.025;
	}

	// Store the value
	this->setValue(oldValue + changeValue);

	// Notify listener
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this, event));
	}

	// Now update if needed
	this->update();
}

//	===========================================================================
void CCenterSlider::handleMouseMovement(CMouseEvent &event)
{
	if (m_isDragging)
	{
		this->handleMovement(event);

		// Notify listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		// Set the cursor
	}
	this->update();
}

//	===========================================================================
void CCenterSlider::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	// Now draw the slider
	CAlphaColour theColour = (m_isDragging) ? m_handleDownColour : m_handleColour;

	// Draw the box in
	graphics.getMutableBrush()->setColour(theColour);
	graphics.fillRectangle(m_sliderArea);
	
	// Draw the surround in
	graphics.getMutablePen()->setColour(m_frameColour);
	graphics.drawRectangle(m_sliderArea);

	// Draw the center line
	graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_RED);
	if (m_isVertical)
	{
		graphics.drawLine(CPoint(0, m_centerPoint), CPoint(m_area.getWidth(), m_centerPoint));
	}
	else
	{
		graphics.drawLine(CPoint(m_centerPoint, 0), CPoint(m_centerPoint, m_area.getHeight()));
	}
}

//	===========================================================================
void CCenterSlider::handleMovement(CMouseEvent &event)
{
	if (m_isVertical)
	{
		if (event.getMousePosition().getYPosition() <= m_centerPoint)
		{
			long top    = event.getMousePosition().getYPosition(); 
			long height = m_centerPoint - top;
			if (top < 0)
			{
				top = 0;
				height = m_centerPoint;
			}
			m_sliderArea.setTop(top);
			m_sliderArea.setHeight(height);

			m_value = 1.0 - ((double)(m_sliderArea.getTop()) / (double)(m_area.getHeight() - m_sliderArea.getHeight()));
		}
		else
		{
			m_sliderArea.setTop(m_centerPoint);
			long height = event.getMousePosition().getYPosition() - m_centerPoint;
			if (height > m_centerPoint)
			{
				height = m_centerPoint;
			}
			m_sliderArea.setHeight(height);

			m_value = 0.5 * (1.0 - ((double)(m_sliderArea.getBottom() - m_centerPoint) / (double)(m_centerPoint)));
		}
	}
	else
	{
		if (event.getMousePosition().getXPosition() <= m_centerPoint)
		{
			// Setting the left
			long left  = event.getMousePosition().getXPosition();
			long width = m_centerPoint - left;
			if (left < 0)
			{
				left  = 0;
				width = m_centerPoint;
			}
			m_sliderArea.setLeft(left);
			m_sliderArea.setWidth(width);
			m_value = ((double)(m_sliderArea.getLeft()) / (double)(m_area.getWidth() - m_sliderArea.getWidth()));
		}
		else
		{
			// Setting the right
			m_sliderArea.setLeft(m_centerPoint);
			long width = event.getMousePosition().getXPosition() - m_centerPoint;
			if (width > m_centerPoint)
			{
				width = m_centerPoint;
			}
			m_sliderArea.setWidth(width);
			m_value = 0.5 + (0.5 * ((double)(m_sliderArea.getRight() - m_centerPoint) / (double)(m_centerPoint)));
		}
	}
}